#region References

using System;
using System.Data;
using System.Data.SqlClient;
using System.Text;
using SProc = gov.va.med.vbecs.Common.VbecsStoredProcs;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;

#endregion

namespace gov.va.med.vbecs.DAL.HL7AL
{
	#region Header

	//<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	//<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	//<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	//<Developers>
	//	<Developer>David Askew</Developer>
	//</Developers>
	//<SiteName>Hines OIFO</SiteName>
	//<CreationDate>5/30/2008</CreationDate>
	//<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	//<summary></summary>

	#endregion

	/// <summary>
	/// Used for handling patient location updates
	/// CR 2960
	/// </summary>
	public sealed class PatientLocation 
	{
        /// <summary>
        /// No location information available
        /// </summary>
        public const string NO_LOCATION_INFORMATION = "No location information is available for selected patient.";
        /// <summary>
        /// Location tooltip base 
        /// </summary>
        public const string LOCATION_TOOLTIP_BASE = "Patient's current location. Class ('I' = Inpatient, 'O' = Outpatient), Point of Care, Room and Bed, Division Code and Division Name.";

        // CR 2522: Room and Bed are now stored in one field
        private const string NEWLINE = "\n"; // CR 3028
        private const string NOT_AVAILABLE = "N/A";
        private const string INPATIENT = "I";
        private const string OUTPATIENT = "O";
        private const string SEPARATOR = ", ";

		#region Variables

		private static char _space = ' ';

		private bool _inPatientIndicator;

		private string _name;
		//
		// CR 2970: Room and Bed are now stored in one field
		private string _roomBed;
		private string _ien;
		private string _divisionCode;
		private string _divisionName;

		private Guid _patientGuid;

		#endregion

		#region Properties

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/30/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8797"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8798"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// DivisionCode
		/// </summary>
		public string DivisionCode
		{
			get
			{
				return _divisionCode;
			}
			set
			{
				_divisionCode = value;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/30/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8799"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8800"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// DivisionName
		/// </summary>
		public string DivisionName
		{
			get
			{
				return _divisionName;
			}
			set
			{
				_divisionName = value;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/30/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8795"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8796"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Ien
		/// </summary>
		public string Ien
		{
			get
			{
				return _ien;
			}
			set
			{
				_ien = value;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/5/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8805"> 
		///		<ExpectedInput>true</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8806"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// InPatientIndicator
		/// </summary>
		public bool InPatientIndicator
		{
			get
			{
				return _inPatientIndicator;
			}
			set
			{
				_inPatientIndicator = value;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/30/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8789"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8790"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Name
		/// </summary>
		public string Name
		{
			get
			{
				return _name;
			}
			set
			{
				_name = value;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/30/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8801"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8802"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// PatientGuid
		/// </summary>
		public Guid PatientGuid
		{
			get
			{
				return _patientGuid;
			}
			set
			{
				_patientGuid = value;
			}
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/30/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8791"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8792"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CR 2970: Room and Bed are now stored in one field
		/// </summary>
		public string RoomBed
		{
			get
			{
				return _roomBed;
			}
			set
			{
				_roomBed = value;
			}
		}

		#endregion

		#region Public Methods

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/30/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8787"> 
		///		<ExpectedInput>Valid PatientGuid</ExpectedInput>
		///		<ExpectedOutput>PatientLocation object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8788"> 
		///		<ExpectedInput>Invalid PatientGuid</ExpectedInput>
		///		<ExpectedOutput>null</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///<summary>
		/// CR 2960, CR 2986 
		/// Returns the Patient's Location in a PatientLocation object;
		/// will return data from PatientTreatment if patient does not
		/// have entry in the PatientLocation table.
		///</summary>
		public static PatientLocation GetCurrentPatientLocation( Guid patientGuid )
		{
			return GetLocation( patientGuid, SProc.GetCurrentPatientLocation.StoredProcName );
		}

        /// <summary>
        /// Returns the Patient's Location name
        /// </summary>
        /// <param name="patientLocation"></param>
        /// <returns></returns>
        public static string GetName(PatientLocation patientLocation)
        {
            if (patientLocation == null)
            {
                return "";
            }
            else
            {
                return patientLocation.Name;
            }
        }

        /// <summary>
        /// Returns the Patient's Location name
        /// CR 2472
        /// </summary>
        /// <param name="patientLocation"></param>
        /// <returns></returns>
        public static string GetToolTip(PatientLocation patientLocation)
        {
            StringBuilder locationBuilder = new StringBuilder();

            if (patientLocation == null)
            {
                locationBuilder.Append(NO_LOCATION_INFORMATION);
            }
            else
            {

                locationBuilder.Append(NEWLINE); // CR 3028
                locationBuilder.Append(patientLocation.InPatientIndicator ? INPATIENT : OUTPATIENT);
                //
                locationBuilder.Append(SEPARATOR);
                //
                if (patientLocation.Name != null && patientLocation.Name.Length > 0)
                {
                    locationBuilder.Append(patientLocation.Name);
                }
                else
                {
                    locationBuilder.Append(NOT_AVAILABLE);
                }
                //
                locationBuilder.Append(SEPARATOR);
                //
                // CR 2522: Room and Bed are now stored in one field
                if (patientLocation.RoomBed != null && patientLocation.RoomBed.Length > 0)
                {
                    locationBuilder.Append(patientLocation.RoomBed);
                }
                else
                {
                    locationBuilder.Append(NOT_AVAILABLE);
                }
                //
                locationBuilder.Append(SEPARATOR);
                //
                if (patientLocation.DivisionCode != null && patientLocation.DivisionCode.Length > 0)
                {
                    locationBuilder.Append(patientLocation.DivisionCode);
                }
                else
                {
                    locationBuilder.Append(NOT_AVAILABLE);
                }
                //
                locationBuilder.Append(SEPARATOR);
                //
                if (patientLocation.DivisionName != null && patientLocation.DivisionName.Length > 0)
                {
                    locationBuilder.Append(patientLocation.DivisionName);
                }
                else
                {
                    locationBuilder.Append(NOT_AVAILABLE);
                }                
            }

            return string.Concat(LOCATION_TOOLTIP_BASE, locationBuilder.ToString());
        }

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/5/2009</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8903"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8904"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		/// CR 2986
		/// Returns the Patient's Location in a PatientLocation object,
		/// using only data in the PatientLocation table (if any)
		///</summary>
		public static PatientLocation GetHL7PatientLocation( Guid patientGuid )
		{
			return GetLocation( patientGuid, SProc.HL7GetPatientLocation.StoredProcName );
		}

		#endregion

		#region Private Methods

		///<summary>
		/// CR 2986
		/// Returns the Patients Location in a PatientLocation object
		///</summary>
		private static PatientLocation GetLocation( Guid patientGuid, string sprocName )
		{
			SqlParameter [] prms =
			{
				new SqlParameter( SProc.GetCurrentPatientLocation.PatientGuid, System.Data.SqlDbType.UniqueIdentifier ),
			};
			//
			prms[0].Value = patientGuid;
			//
			DataTable locationTable = Common.StoredProcedure.GetDataTable( sprocName , prms );
			//
			DataRow locationRow = null;
			//
			if ( locationTable != null && locationTable.Rows != null && locationTable.Rows.Count == 1 && locationTable.Rows[0] != null )
			{
				locationRow = locationTable.Rows[0];
			}
			//
			if ( locationRow != null )
			{ 
				PatientLocation patientLocation = new PatientLocation();
				//
				patientLocation.PatientGuid = patientGuid;
				patientLocation.DivisionCode = Convert.ToString( locationRow[TABLE.PatientLocation.DivisionCode] ).TrimEnd(_space);
				patientLocation.InPatientIndicator = locationRow.IsNull(TABLE.PatientLocation.InPatientIndicator) ? false : Convert.ToBoolean( locationRow[TABLE.PatientLocation.InPatientIndicator] );
				patientLocation.Name = Convert.ToString( locationRow[TABLE.PatientLocation.LocationName] ).TrimEnd(_space);
				//
				// CR 2970: Room and Bed are now stored in one field
				patientLocation.RoomBed = Convert.ToString( locationRow[TABLE.PatientLocation.LocationRoomBed] ).TrimEnd(_space);
				//
				// CR 3039: Adding DivisionName and LocationIen (if available)
				if ( locationRow.Table.Columns.Contains(TABLE.VamcDivision.DivisionName) )
				{
					patientLocation.DivisionName = Convert.ToString(locationRow[TABLE.VamcDivision.DivisionName]).TrimEnd(_space); 
				}
				if ( locationRow.Table.Columns.Contains(TABLE.HospitalLocationCache.LocationIen) )
				{
					patientLocation.Ien = Convert.ToString(locationRow[TABLE.HospitalLocationCache.LocationIen]).TrimEnd(_space);  
				}
				//
				return patientLocation;
			}
			//
			return null;
		}

		/// <summary>
		/// CR 2960
		/// </summary>
		public static DataTable GetPatientLocationDataTable()
		{
			DataTable dtPatientLocation = new DataTable("PatientLocation");
			//
			dtPatientLocation.Columns.Add(TABLE.PatientLocation.PatientGuid, typeof(System.Guid));
			dtPatientLocation.Columns.Add(TABLE.PatientLocation.InPatientIndicator, typeof(bool));
			dtPatientLocation.Columns.Add(TABLE.PatientLocation.LocationName, typeof(string));
			//
			// CR 2970: Room and Bed are now stored in one field
			dtPatientLocation.Columns.Add(TABLE.PatientLocation.LocationRoomBed, typeof(string));
			dtPatientLocation.Columns.Add(TABLE.PatientLocation.DivisionCode, typeof(string));
			dtPatientLocation.Columns.Add(TABLE.PatientLocation.LastUpdateFunctionId, typeof(int));
			dtPatientLocation.Columns.Add(TABLE.PatientLocation.LastUpdateUser, typeof(string));
			//
			return dtPatientLocation;
		}	

		#endregion
	}
}
